﻿/* 
 Name:
	SplitImage
 Version:
	2.2 (2 Nov 2011) by Lloyd Alvarez
      Changelog:
	Changed app.language to app.isoLanguage due to Adobe deprecation
 Author:
	Charles Bordenave
 Description:  
	This script splits the selected project item into rectangular pieces. 
 Usage:
	In After Effects CS4 or later, run the script
	Select a project item
	Specify the number of columns and rows  
	Click on Execute
*/


// This class represents the main class of the script
function SplitImage()
{
	// Variable used to keep track of 'this' reference
	var splitImage = this;
	
	// Create an instance of the utils class to use its functions
	var utils = new SplitImageUtils();

	// Script infos
	this.scriptMinSupportVersion = "9.0";
	this.scriptName = "SplitImage.jsx";	
	this.scriptVersion = "2.2";
	this.scriptTitle = "Split Image";
	this.scriptCopyright = "Copyright (c) 2010 Charles Bordenave";
	this.scriptHomepage = "http://www.nabscripts.com";
	this.scriptDescription = {en:"This script splits the selected project item into rectangular pieces.", 
							  fr:"Ce script divise l\\'item sélectionné dans le panneau de projet en parties rectangulaires."};	
	this.scriptAbout = {en:this.scriptName + ", v" + this.scriptVersion + "\\r" + this.scriptCopyright + "\\r" + this.scriptHomepage + "\\r\\r" + utils.loc(this.scriptDescription), 
						fr:this.scriptName + ", v" + this.scriptVersion + "\\r" + this.scriptCopyright + "\\r" + this.scriptHomepage + "\\r\\r" + utils.loc(this.scriptDescription)};		
	this.scriptUsage = {en:	"\u25BA In After Effects CS4 or later, run the script \\r" +
							"\u25BA Select a project item \\r" +
							"\u25BA Specify the number of columns and rows \\r" + 
							"\u25BA Click on Execute",
						fr:	"\u25BA Dans After Effects CS4 ou supérieur, exécuter le script \\r" +
							"\u25BA Sélectionner un item dans le panneau de projet \\r" +
							"\u25BA Spécifier le nombre de colonnes et de lignes \\r" +
							"\u25BA Cliquer sur Exécuter"}; 
							
	// Errors
	this.requirementErr = {en:"This script requires After Effects CS4 or later.", fr:"Ce script nécessite After Effects CS4 ou supérieur."};	
	this.noItemErr = {en:"Select a project item.", fr:"Sélectionnez un item dans le panneau projet."};

	// UI strings & default settings 
	this.aboutBtnName = "?";
	this.columnsStName = {en:"Columns:", fr:"Colonnes:"};
	this.columnsEtDflt = 4;
	this.rowsStName = {en:"Rows:", fr:"Lignes:"};
	this.rowsEtDflt = 3;
	this.runBtnName = {en:"Execute", fr:"Exécuter"};
	
	// Miscellaneous
	this.tilesFolderName = {en:"Tiles", fr:"Morceaux"};
	this.stillDurationDflt = 30; // in seconds
	this.stillFrameRateDflt = 30; // in frame per seconds
	
	// Creates and displays the script interface
	this.buildUI = function (thisObj)
	{
		// dockable panel or palette
		var pal = (thisObj instanceof Panel) ? thisObj : new Window("palette", this.scriptTitle, undefined, {resizeable:false});

		// resource specifications
		var res =
		"group { orientation:'column', alignment:['left','top'], alignChildren:['right','top'], \
			gr1: Group { \
				aboutBtn: Button { text:'" + this.aboutBtnName + "', preferredSize:[25,20] } \
			}, \
			gr2: Group { \
				gr21: Group { \
					columnsSt: StaticText { text:'" + utils.loc(this.columnsStName) + "' }, \
					columnsEt: EditText { text:'" + this.columnsEtDflt + "', characters:5 } \
				}, \
				gr22: Group { \
					rowsSt: StaticText { text:'" + utils.loc(this.rowsStName) + "' }, \
					rowsEt: EditText { text:'" + this.rowsEtDflt + "', characters:5 } \
				} \
			}, \
			gr3: Panel { alignment:['fill','center'] }, \
			gr4: Group { \
				runBtn: Button { text:'" + utils.loc(this.runBtnName) + "' } \
			} \
		}"; 
		pal.gr = pal.add(res);
		
		// event callbacks
		pal.gr.gr1.aboutBtn.onClick = function () 
		{ 
			utils.createAboutDlg(splitImage.scriptAbout, splitImage.scriptUsage); 
		};
		
		pal.gr.gr2.gr21.columnsEt.onChange = function () 
		{ 
			if (isNaN(this.text)) this.text = splitImage.columnsEtDflt;
			else this.text = parseInt(this.text);
		};

		pal.gr.gr2.gr22.rowsEt.onChange = function () 
		{ 
			if (isNaN(this.text)) this.text = splitImage.rowsEtDflt;
			else this.text = parseInt(this.text);
		};

		pal.gr.gr4.runBtn.onClick = function () 
		{ 
		   splitImage.split(pal); 
		};
				
		// show user interface
		if (pal instanceof Window)
		{
			pal.center();
			pal.show();
		}
		else
		{
			pal.layout.layout(true);
		}	   
	};

	// Functional part of the script: splits the selected layer into rectangular pieces
	this.split = function (pal)
	{
		try
		{
			var selItem = app.project.selection[0];
			var err = this.noItemErr;
			if (!selItem) throw(err);
			
			var columns = parseFloat(pal.gr.gr2.gr21.columnsEt.text);
			var rows = parseFloat(pal.gr.gr2.gr22.rowsEt.text);
			
			var W = selItem.width;
			var H = selItem.height;		
			var tileW = Math.floor(W/columns);//(layer.source.pixelAspect/comp.pixelAspect));
			var tileH = Math.floor(H/rows);
			var numTiles = columns*rows;				
			var dur = selItem.duration > 0 ? selItem.duration : this.stillDurationDflt;
			var fr = selItem.frameRate > 0 ? selItem.frameRate : this.stillFrameRateDflt;
			
			app.beginUndoGroup(this.scriptTitle);
			
			var tilesFolder = app.project.items.addFolder(selItem.name + " - " + utils.loc(this.tilesFolderName));
			
			var comp = app.project.items.addComp(selItem.name + " - Splitted", selItem.width, selItem.height, selItem.pixelAspect, dur, fr);
			
			for (var i = 0; i < numTiles; i++) 
			{
				var r = 1 + Math.floor(i/columns);
				var c = 1 + i%columns;
				var suffix = " - " + r + "," + c;
				var preCompName = selItem.name.substring(0,20) + suffix;
				
				var preComp = app.project.items.addComp(preCompName, tileW, tileH, selItem.pixelAspect, dur, fr);			
				preComp.parentFolder = tilesFolder;
				
				var preCompLayer = preComp.layers.add(selItem);
				
				var x = (i%columns)*tileW; 
				var y = Math.floor(i/columns)*tileH;			
				var newPos = [W/2,H/2] - [x,y];
							
				preCompLayer.position.setValueAtTime(0, newPos);
						
				var tileLayer = comp.layers.add(preComp);
				
				var pos = tileLayer.position.value;
				var xPos = pos[0] - W/2 + tileW/2 + (i%columns)*tileW;
				var yPos = pos[1] - H/2 + tileH/2 + Math.floor(i/columns)*tileH;
				var zPos = pos[2];
				
				tileLayer.position.setValue([xPos, yPos, zPos]);								
				tileLayer.moveToEnd();
				tileLayer.selected = false;
			}			
			
			selItem.selected = false;
			comp.selected = true;
	  
			app.endUndoGroup();
		}
		catch(err)
		{
			utils.throwErr(err);
		}				
	};
	
	// Runs the script  
	this.run = function (thisObj) 
	{
		if (parseFloat(app.version) < parseFloat(this.scriptMinSupportVersion))
		{
			utils.throwErr(this.requirementErr);
		}
		else
		{
			this.buildUI(thisObj);
		}	
	};
}


// This class provides some utility functions
function SplitImageUtils()
{
	// Variable used to keep track of 'this' reference
	var utils = this;
	
	// String localization function: english and french languages are supported
	this.loc = function (str)
	{
		  if (parseFloat(app.version) < 9 )
                var localLang = $.locale;
                else var localLang = app.isoLanguage;
		return localLang.toLowerCase().match("fr") ? str.fr : str.en;
	};

	// Displays a window containg a localized error message
	this.throwErr = function (err)
	{
		var wndTitle = $.fileName.substring($.fileName.lastIndexOf("/")+1, $.fileName.lastIndexOf("."));
		Window.alert("Script error:\r" + this.loc(err), wndTitle, true);
	};			

	// Displays a dialog containg the script description and usage
	this.createAboutDlg = function (aboutStr, usageStr)
	{	
		eval(unescape('%09%09%76%61%72%20%64%6C%67%20%3D%20%6E%65%77%20%57%69%6E%64%6F%77%28%22%64%69%61%6C%6F%67%22%2C%20%22%41%62%6F%75%74%22%29%3B%0A%09%20%20%20%20%20%20%09%20%20%20%20%20%20%20%09%0A%09%20%20%20%20%76%61%72%20%72%65%73%20%3D%0A%09%09%22%67%72%6F%75%70%20%7B%20%6F%72%69%65%6E%74%61%74%69%6F%6E%3A%27%63%6F%6C%75%6D%6E%27%2C%20%61%6C%69%67%6E%6D%65%6E%74%3A%5B%27%66%69%6C%6C%27%2C%27%66%69%6C%6C%27%5D%2C%20%61%6C%69%67%6E%43%68%69%6C%64%72%65%6E%3A%5B%27%66%69%6C%6C%27%2C%27%66%69%6C%6C%27%5D%2C%20%5C%0A%09%09%09%70%6E%6C%3A%20%50%61%6E%65%6C%20%7B%20%74%79%70%65%3A%27%74%61%62%62%65%64%70%61%6E%65%6C%27%2C%20%5C%0A%09%09%09%09%61%62%6F%75%74%54%61%62%3A%20%50%61%6E%65%6C%20%7B%20%74%79%70%65%3A%27%74%61%62%27%2C%20%74%65%78%74%3A%27%44%65%73%63%72%69%70%74%69%6F%6E%27%2C%20%5C%0A%09%09%09%09%09%61%62%6F%75%74%45%74%3A%20%45%64%69%74%54%65%78%74%20%7B%20%74%65%78%74%3A%27%22%20%2B%20%74%68%69%73%2E%6C%6F%63%28%61%62%6F%75%74%53%74%72%29%20%2B%20%22%27%2C%20%70%72%65%66%65%72%72%65%64%53%69%7A%65%3A%5B%33%36%30%2C%32%30%30%5D%2C%20%70%72%6F%70%65%72%74%69%65%73%3A%7B%6D%75%6C%74%69%6C%69%6E%65%3A%74%72%75%65%7D%20%7D%20%5C%0A%09%09%09%09%7D%2C%20%5C%0A%09%09%09%09%75%73%61%67%65%54%61%62%3A%20%50%61%6E%65%6C%20%7B%20%74%79%70%65%3A%27%74%61%62%27%2C%20%74%65%78%74%3A%27%55%73%61%67%65%27%2C%20%5C%0A%09%09%09%09%09%75%73%61%67%65%45%74%3A%20%45%64%69%74%54%65%78%74%20%7B%20%74%65%78%74%3A%27%22%20%2B%20%74%68%69%73%2E%6C%6F%63%28%75%73%61%67%65%53%74%72%29%20%2B%20%22%27%2C%20%70%72%65%66%65%72%72%65%64%53%69%7A%65%3A%5B%33%36%30%2C%32%30%30%5D%2C%20%70%72%6F%70%65%72%74%69%65%73%3A%7B%6D%75%6C%74%69%6C%69%6E%65%3A%74%72%75%65%7D%20%7D%20%5C%0A%09%09%09%09%7D%20%5C%0A%09%09%09%7D%2C%20%5C%0A%09%09%09%62%74%6E%73%3A%20%47%72%6F%75%70%20%7B%20%6F%72%69%65%6E%74%61%74%69%6F%6E%3A%27%72%6F%77%27%2C%20%61%6C%69%67%6E%6D%65%6E%74%3A%5B%27%66%69%6C%6C%27%2C%27%62%6F%74%74%6F%6D%27%5D%2C%20%5C%0A%09%09%09%09%6F%74%68%65%72%53%63%72%69%70%74%73%42%74%6E%3A%20%42%75%74%74%6F%6E%20%7B%20%74%65%78%74%3A%27%4F%74%68%65%72%20%53%63%72%69%70%74%73%2E%2E%2E%27%2C%20%61%6C%69%67%6E%6D%65%6E%74%3A%5B%27%6C%65%66%74%27%2C%27%63%65%6E%74%65%72%27%5D%20%7D%2C%20%5C%0A%09%09%09%09%6F%6B%42%74%6E%3A%20%42%75%74%74%6F%6E%20%7B%20%74%65%78%74%3A%27%4F%6B%27%2C%20%61%6C%69%67%6E%6D%65%6E%74%3A%5B%27%72%69%67%68%74%27%2C%27%63%65%6E%74%65%72%27%5D%20%7D%20%5C%0A%09%09%09%7D%20%5C%0A%09%09%7D%22%3B%20%0A%09%09%64%6C%67%2E%67%72%20%3D%20%64%6C%67%2E%61%64%64%28%72%65%73%29%3B%0A%09%09%0A%09%09%64%6C%67%2E%67%72%2E%70%6E%6C%2E%61%62%6F%75%74%54%61%62%2E%61%62%6F%75%74%45%74%2E%6F%6E%43%68%61%6E%67%65%20%3D%20%64%6C%67%2E%67%72%2E%70%6E%6C%2E%61%62%6F%75%74%54%61%62%2E%61%62%6F%75%74%45%74%2E%6F%6E%43%68%61%6E%67%69%6E%67%20%3D%20%66%75%6E%63%74%69%6F%6E%20%28%29%0A%09%09%7B%0A%09%09%09%74%68%69%73%2E%74%65%78%74%20%3D%20%75%74%69%6C%73%2E%6C%6F%63%28%61%62%6F%75%74%53%74%72%29%2E%72%65%70%6C%61%63%65%28%2F%5C%5C%72%2F%67%2C%20%27%5C%72%27%29%3B%0A%09%09%7D%3B%0A%09%09%0A%09%09%64%6C%67%2E%67%72%2E%70%6E%6C%2E%75%73%61%67%65%54%61%62%2E%75%73%61%67%65%45%74%2E%6F%6E%43%68%61%6E%67%65%20%3D%20%64%6C%67%2E%67%72%2E%70%6E%6C%2E%75%73%61%67%65%54%61%62%2E%75%73%61%67%65%45%74%2E%6F%6E%43%68%61%6E%67%69%6E%67%20%3D%20%66%75%6E%63%74%69%6F%6E%20%28%29%0A%09%09%7B%0A%09%09%09%74%68%69%73%2E%74%65%78%74%20%3D%20%75%74%69%6C%73%2E%6C%6F%63%28%75%73%61%67%65%53%74%72%29%2E%72%65%70%6C%61%63%65%28%2F%5C%5C%72%2F%67%2C%20%27%5C%72%27%29%2E%72%65%70%6C%61%63%65%28%2F%5C%5C%27%2F%67%2C%20%22%27%22%29%3B%0A%09%09%7D%3B%0A%09%09%09%0A%09%09%64%6C%67%2E%67%72%2E%62%74%6E%73%2E%6F%74%68%65%72%53%63%72%69%70%74%73%42%74%6E%2E%6F%6E%43%6C%69%63%6B%20%3D%20%66%75%6E%63%74%69%6F%6E%20%28%29%0A%09%09%7B%0A%09%09%09%76%61%72%20%63%6D%64%20%3D%20%22%22%3B%0A%09%09%09%76%61%72%20%75%72%6C%20%3D%20%22%68%74%74%70%3A%2F%2F%61%65%73%63%72%69%70%74%73%2E%63%6F%6D%2F%63%61%74%65%67%6F%72%79%2F%73%63%72%69%70%74%73%2F%6E%61%62%2F%22%3B%0A%09%0A%09%09%09%69%66%20%28%24%2E%6F%73%2E%69%6E%64%65%78%4F%66%28%22%57%69%6E%22%29%20%21%3D%20%2D%31%29%0A%09%09%09%7B%0A%09%20%20%20%20%20%20%20%20%09%69%66%20%28%46%69%6C%65%28%22%43%3A%2F%50%72%6F%67%72%61%6D%20%46%69%6C%65%73%2F%4D%6F%7A%69%6C%6C%61%20%46%69%72%65%66%6F%78%2F%66%69%72%65%66%6F%78%2E%65%78%65%22%29%2E%65%78%69%73%74%73%29%0A%09%09%09%09%09%63%6D%64%20%2B%3D%20%22%43%3A%2F%50%72%6F%67%72%61%6D%20%46%69%6C%65%73%2F%4D%6F%7A%69%6C%6C%61%20%46%69%72%65%66%6F%78%2F%66%69%72%65%66%6F%78%2E%65%78%65%20%22%20%2B%20%75%72%6C%3B%0A%09%09%09%09%65%6C%73%65%20%69%66%20%28%46%69%6C%65%28%22%43%3A%2F%50%72%6F%67%72%61%6D%20%46%69%6C%65%73%20%28%78%38%36%29%2F%4D%6F%7A%69%6C%6C%61%20%46%69%72%65%66%6F%78%2F%66%69%72%65%66%6F%78%2E%65%78%65%22%29%2E%65%78%69%73%74%73%29%0A%09%09%09%09%09%63%6D%64%20%2B%3D%20%22%43%3A%2F%50%72%6F%67%72%61%6D%20%46%69%6C%65%73%20%28%78%38%36%29%2F%4D%6F%7A%69%6C%6C%61%20%46%69%72%65%66%6F%78%2F%66%69%72%65%66%6F%78%2E%65%78%65%20%22%20%2B%20%75%72%6C%3B%0A%09%09%09%09%65%6C%73%65%0A%09%09%09%09%09%63%6D%64%20%2B%3D%20%22%43%3A%2F%50%72%6F%67%72%61%6D%20%46%69%6C%65%73%2F%49%6E%74%65%72%6E%65%74%20%45%78%70%6C%6F%72%65%72%2F%69%65%78%70%6C%6F%72%65%2E%65%78%65%20%22%20%2B%20%75%72%6C%3B%0A%09%09%09%7D%0A%09%09%09%65%6C%73%65%0A%09%09%09%09%63%6D%64%20%2B%3D%20%22%6F%70%65%6E%20%5C%22%22%20%2B%20%75%72%6C%20%2B%20%22%5C%22%22%3B%20%20%20%20%20%20%20%20%20%0A%09%09%09%74%72%79%0A%09%09%09%7B%0A%09%09%09%09%73%79%73%74%65%6D%2E%63%61%6C%6C%53%79%73%74%65%6D%28%63%6D%64%29%3B%0A%09%09%09%7D%0A%09%09%09%63%61%74%63%68%28%65%29%0A%09%09%09%7B%0A%09%09%09%09%61%6C%65%72%74%28%65%29%3B%0A%09%09%09%7D%0A%09%09%7D%3B%0A%09%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%0A%09%09%64%6C%67%2E%67%72%2E%62%74%6E%73%2E%6F%6B%42%74%6E%2E%6F%6E%43%6C%69%63%6B%20%3D%20%66%75%6E%63%74%69%6F%6E%20%28%29%20%0A%09%09%7B%0A%09%09%09%64%6C%67%2E%63%6C%6F%73%65%28%29%3B%20%0A%09%09%7D%3B%0A%09%20%20%20%20%20%20%20%0A%09%09%64%6C%67%2E%63%65%6E%74%65%72%28%29%3B%0A%09%09%64%6C%67%2E%73%68%6F%77%28%29%3B'));		
	};
}


// Creates an instance of the main class and run it
new SplitImage().run(this);
